<?php
session_start();
require '../assets/db_connect.php';
require '../libraries/php-qrcode-main/qrlib.php'; // adjust path if needed

// -------------------------------
// 1. Validate sale ID
// -------------------------------
if (!isset($_GET['id'])) {
    echo "<div style='color:red;'>Sale ID missing.</div>";
    exit;
}

$sale_id = intval($_GET['id']);

// -------------------------------
// 2. Fetch sale + shop info
// -------------------------------
$stmt = $pdo->prepare("
    SELECT 
        s.*,
        c.name AS customer_name,
        u.username AS created_by_name,
        sh.name AS shop_name,
        sh.phone AS shop_phone,
        sh.address AS shop_address
    FROM sales s
    LEFT JOIN customers c ON s.customer_id = c.id
    LEFT JOIN users u ON s.created_by = u.id
    LEFT JOIN shops sh ON s.shop_id = sh.id
    WHERE s.id = ?
");
$stmt->execute([$sale_id]);
$sale = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$sale) {
    echo "<div style='color:red;'>Sale not found.</div>";
    exit;
}

// -------------------------------
// 3. Fetch sale items
// -------------------------------
$stmt_items = $pdo->prepare("
    SELECT sl.*, p.name AS product_name, b.batch_no
    FROM sale_lines sl
    LEFT JOIN products p ON sl.product_id = p.id
    LEFT JOIN product_batches b ON sl.batch_id = b.id
    WHERE sl.sale_id = ?
");
$stmt_items->execute([$sale_id]);
$items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);

// -------------------------------
// 4. Generate QR Code
// -------------------------------
$qrTempDir = "../temp_qr/";
if (!is_dir($qrTempDir)) {
    mkdir($qrTempDir, 0777, true);
}

$qrFilename = $qrTempDir . $sale['invoice_no'] . ".png";

$qrData = "SHOP: {$sale['shop_name']}
Invoice: {$sale['invoice_no']}
Date: {$sale['sale_date']}
Total: {$sale['total']}";

QRcode::png($qrData, $qrFilename, QR_ECLEVEL_L, 3);
?>
<!DOCTYPE html>
<html>
<head>
    <title>Invoice <?= htmlspecialchars($sale['invoice_no']) ?></title>

    <style>
        body {
            font-family: 'Courier New', monospace;
            font-size: 12px;
        }

        @media print {
            body {
                width: 58mm; /* Change to 80mm if needed */
                margin: 0;
                padding: 0;
            }
        }

        .center { text-align: center; }
        table { width: 100%; border-collapse: collapse; margin-top: 5px; }
        th, td { padding: 2px 0; border-bottom: 1px dashed #000; }
        .qr img { width: 100px; margin-top: 5px; }
    </style>
</head>

<body onload="window.print(); setTimeout(() => window.close(), 900);">

<div class="receipt">

    <!-- SHOP HEADER -->
    <div class="center">
        <h3><?= strtoupper(htmlspecialchars($sale['shop_name'])) ?></h3>
        <?php if (!empty($sale['shop_address'])): ?>
            <p><?= htmlspecialchars($sale['shop_address']) ?></p>
        <?php endif; ?>
        <p>Tel: <?= htmlspecialchars($sale['shop_phone']) ?></p>
        <hr>
    </div>

    <!-- SALE DETAILS -->
    <p><b>Invoice:</b> <?= htmlspecialchars($sale['invoice_no']) ?></p>
    <p><b>Date:</b> <?= date("d-M-Y H:i", strtotime($sale['sale_date'])) ?></p>
    <p><b>Cashier:</b> <?= htmlspecialchars($sale['created_by_name']) ?></p>

    <table>
        <thead>
            <tr>
                <th>Item</th>
                <th>Q</th>
                <th>P</th>
                <th>T</th>
            </tr>
        </thead>

        <tbody>
            <?php foreach ($items as $it): ?>
            <tr>
                <td><?= htmlspecialchars($it['product_name']) ?></td>
                <td><?= $it['qty'] ?></td>
                <td><?= number_format($it['unit_price'], 2) ?></td>
                <td><?= number_format($it['line_total'], 2) ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <hr>

    <!-- TOTALS -->
    <p><b>Total:</b> <?= number_format($sale['total'],2) ?></p>
    <p><b>Paid:</b> <?= number_format($sale['paid'],2) ?></p>
    <p><b>Balance:</b> <?= number_format($sale['total'] - $sale['paid'],2) ?></p>

    <!-- QR CODE -->
    <div class="center qr">
        <img src="<?= $qrFilename ?>" alt="QR Code">
        <p>Scan QR</p>
    </div>

    <!-- FOOTER -->
    <div class="center">
        <hr>
        <p>*** THANK YOU ***</p>
    </div>

</div>

</body>
</html>
