<?php
session_start();
require_once '../assets/db_connect.php';
require_once '../includes/header.php';
require_once '../includes/sidebar.php';

if (!isset($_SESSION['user'])) { 
    echo "Access denied"; 
    exit; 
}

$user_role = $_SESSION['user']['role'];
$user_shop_id = $_SESSION['user']['shop_id'];

$from = $_GET['from'] ?? date('Y-m-01');
$to   = $_GET['to']   ?? date('Y-m-d');

// ADMIN → choose shop
if ($user_role === 'admin') {
    $shop_id = $_GET['shop_id'] ?? 0;
} else {
    $shop_id = $user_shop_id;
}

/* ---- SQL ---- */
$sql = "
 SELECT s.id, s.invoice_no, s.sale_date,
        COALESCE(c.name,'Walk-in') AS customer,
        s.total, s.paid, s.status,
        u.fullname AS cashier,
        sh.name AS shop_name
 FROM sales s
 LEFT JOIN customers c ON s.customer_id = c.id
 LEFT JOIN users u ON s.created_by = u.id
 LEFT JOIN shops sh ON s.shop_id = sh.id
 WHERE DATE(s.sale_date) BETWEEN ? AND ?
";

$params = [$from, $to];

if ($shop_id != 0) {
    $sql .= " AND s.shop_id = ? ";
    $params[] = $shop_id;
}

$sql .= " ORDER BY s.sale_date DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$sales = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ================================
   TOTALS CALCULATION  (ADDED)
================================ */
$sum_total = 0;
$sum_paid  = 0;

foreach ($sales as $s) {
    $sum_total += $s['total'];
    $sum_paid  += $s['paid'];
}

$sum_balance = $sum_total - $sum_paid;
/* ================================= */




/* ---- CSV EXPORT ---- */
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename=sales_'.$from.'_'.$to.'.csv');
    $out = fopen('php://output','w');
    fputcsv($out, ['id','invoice_no','sale_date','customer','total','paid','status','cashier','shop']);

    foreach ($sales as $s) {
        fputcsv($out, [
            $s['id'], $s['invoice_no'], $s['sale_date'], $s['customer'],
            $s['total'], $s['paid'], $s['status'], $s['cashier'], $s['shop_name']
        ]);
    }
    fclose($out);
    exit;
}
?>

<!-- PRINT CSS -->
<style>
@media print {
    body * { visibility: hidden; }
    #print-area, #print-area * { visibility: visible; }
    #print-area { position: absolute; left: 0; top: 0; width: 100%; }
    .no-print { display: none !important; }
}
</style>

<div class="col-md-10 ml-sm-auto px-4 py-4">

<div class="no-print">
<h3 class="text-success">Sales Between Dates</h3>

<form class="form-inline mb-3">
  <label class="mr-2">From</label>
  <input type="date" name="from" value="<?= $from ?>" class="form-control mr-2">

  <label class="mr-2">To</label>
  <input type="date" name="to" value="<?= $to ?>" class="form-control mr-2">

  <?php if ($user_role === 'admin'): ?>
      <label class="mr-2">Shop</label>
      <select name="shop_id" class="form-control mr-2">
          <option value="0">All Shops</option>
          <?php
              $shops = $pdo->query("SELECT id, name FROM shops ORDER BY name")->fetchAll();
              foreach ($shops as $sh):
          ?>
              <option value="<?= $sh['id'] ?>" <?= ($shop_id == $sh['id'] ? 'selected' : '') ?>>
                  <?= htmlspecialchars($sh['name']) ?>
              </option>
          <?php endforeach; ?>
      </select>
  <?php endif; ?>

  <button class="btn btn-primary mr-2">Filter</button>

  <a href="?from=<?= $from ?>&to=<?= $to ?>&shop_id=<?= $shop_id ?>&export=csv" 
     class="btn btn-outline-secondary mr-2">
     Export CSV
  </a>

  <button type="button" onClick="window.print();" class="btn btn-dark">
     Print
  </button>
</form>
</div>



<!-- PRINT AREA START -->
<div id="print-area">

<h4 class="text-center">
    <strong>Sales Report (<?= date('d M Y', strtotime($from)) ?> — <?= date('d M Y', strtotime($to)) ?>)</strong>
</h4>

<?php if ($shop_id == 0): ?>
    <p class="text-center"><em>All Shops</em></p>
<?php else: ?>
    <p class="text-center"><em>Shop: <?= htmlspecialchars($sales[0]['shop_name'] ?? '') ?></em></p>
<?php endif; ?>


<div class="table-responsive">
<table class="table table-bordered table-sm">
  <thead>
    <tr>
      <th>ID</th>
      <th>Invoice</th>
      <th>Date</th>
      <th>Customer</th>
      <th>Total</th>
      <th>Paid</th>
      <th>Status</th>
      <th>Cashier</th>
      <th>Shop</th>
    </tr>
  </thead>

  <tbody>
    <?php foreach($sales as $s): ?>
      <tr>
        <td><?= $s['id'] ?></td>
        <td><?= htmlspecialchars($s['invoice_no']) ?></td>
        <td><?= date('d-M-Y H:i', strtotime($s['sale_date'])) ?></td>
        <td><?= htmlspecialchars($s['customer']) ?></td>
        <td><?= number_format($s['total'],2) ?></td>
        <td><?= number_format($s['paid'],2) ?></td>
        <td><?= htmlspecialchars($s['status']) ?></td>
        <td><?= htmlspecialchars($s['cashier']) ?></td>
        <td><?= htmlspecialchars($s['shop_name']) ?></td>
      </tr>
    <?php endforeach; ?>
  </tbody>

  <!-- ================================
        TOTALS ROW (ADDED)
  ================================= -->
  <tfoot>
    <tr style="font-weight:bold; background:#f0f0f0;">
      <td colspan="4" class="text-right">TOTALS:</td>
      <td><?= number_format($sum_total,2) ?></td>
      <td><?= number_format($sum_paid,2) ?></td>
      <td><?= number_format($sum_balance,2) ?></td>
      <td colspan="2"></td>
    </tr>
  </tfoot>

</table>
</div>

</div>
<!-- PRINT AREA END -->

</div>

<?php require_once '../includes/footer.php'; ?>
