<?php 
session_start();
require_once '../assets/db_connect.php';
require_once '../includes/header.php';
require_once '../includes/sidebar.php';

if (!isset($_SESSION['user'])) { 
    echo "Access denied"; 
    exit; 
}

$user_role = $_SESSION['user']['role'] ?? '';
$user_shop_id = $_SESSION['user']['shop_id'] ?? 0;

// Get selected date
$date = $_GET['date'] ?? date('Y-m-d');

// ===== SHOP HANDLING =====
if($user_role === 'admin') {
    // Admin can see all shops
    $shops = $pdo->query("SELECT id, name FROM shops ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
    $shop_id = $_GET['shop_id'] ?? 0; // 0 = all shops
} else {
    // Non-admin: only their assigned shop
    $shops = [];
    if($user_shop_id){
        $stmt = $pdo->prepare("SELECT id, name FROM shops WHERE id=?");
        $stmt->execute([$user_shop_id]);
        $shops = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $shop_id = $user_shop_id; // force filter
    } else {
        echo "No shop assigned";
        exit;
    }
}

// ===== FETCH SALES SUMMARY =====
$sql = "
 SELECT
   COUNT(DISTINCT s.id) AS invoices,
   SUM(s.total) AS total_revenue,
   SUM(sl.qty) AS total_qty,
   SUM(sl.unit_price * sl.qty - sl.discount) AS calc_total
 FROM sales s
 JOIN sale_lines sl ON sl.sale_id = s.id
 WHERE DATE(s.sale_date) = ?
";

$params = [$date];

if($shop_id && $shop_id != 0){
    $sql .= " AND s.shop_id = ?";
    $params[] = $shop_id;
}

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$row = $stmt->fetch(PDO::FETCH_ASSOC);

// ===== CSV Export =====
if (isset($_GET['export']) && $_GET['export']==='csv') {
    header('Content-Type: text/csv');
    $filename = 'sales_daily_'.$date.($shop_id?'_shop'.$shop_id:'').'.csv';
    header('Content-Disposition: attachment; filename='.$filename);
    $out = fopen('php://output','w');

    $header = ['date','shop','invoices','total_revenue','total_qty'];
    fputcsv($out, $header);

    $shop_name = ($shop_id && $shop_id!=0) ? ($shops[0]['name'] ?? '') : 'All Shops';

    fputcsv($out, [$date, $shop_name, $row['invoices'] ?? 0, $row['total_revenue'] ?? 0, $row['total_qty'] ?? 0]);
    fclose($out);
    exit;
}
?>
 <style>
    @media print {
    .sidebar, .navbar, .btn, form {
        display: none !important;
    }
    body {
        margin: 0;
        background: white;
    }
    table {
        font-size: 12px;
    }
}

    </style>
<div class="col-md-10 ml-sm-auto px-4 py-4">
  <h3 class="text-success">Daily Sales Summary — <?= htmlspecialchars($date) ?></h3>

  <form class="form-inline mb-3" method="get">
    <label class="mr-2">Date</label>
    <input type="date" name="date" value="<?= htmlspecialchars($date) ?>" onchange="this.form.submit()" class="form-control mr-2">

    <?php if($user_role === 'admin'): ?>
        <label class="mr-2">Shop</label>
        <select name="shop_id" class="form-control mr-2" onchange="this.form.submit()">
            <option value="0">All Shops</option>
            <?php foreach($shops as $s): ?>
                <option value="<?= $s['id'] ?>" <?= $shop_id==$s['id']?'selected':'' ?>><?= htmlspecialchars($s['name']) ?></option>
            <?php endforeach; ?>
        </select>
    <?php else: ?>
        <input type="hidden" name="shop_id" value="<?= $shop_id ?>">
        <span class="mr-2 font-weight-bold"><?= htmlspecialchars($shops[0]['name'] ?? '') ?></span>
    <?php endif; ?>

    <button onclick="window.print()" class="btn btn-primary">
    <i class="fa fa-print"></i> Print
</button>

  </form>

  <table class="table table-bordered">
    <tr><th>Invoices</th><td><?= $row['invoices'] ?? 0 ?></td></tr>
    <tr><th>Total Revenue</th><td><?= number_format($row['total_revenue'] ?? 0,2) ?></td></tr>
    <tr><th>Total Quantity Sold</th><td><?= $row['total_qty'] ?? 0 ?></td></tr>
  </table>
</div>

<?php require_once '../includes/footer.php'; ?>
