<?php
session_start();
require_once '../assets/db_connect.php';
require_once '../includes/header.php';
require_once '../includes/sidebar.php';

if (!isset($_SESSION['user'])) { echo "Access denied"; exit; }

$user_role = $_SESSION['user']['role'];
$user_shop_id = $_SESSION['user']['shop_id'];

$from = $_GET['from'] ?? date('Y-m-01');
$to   = $_GET['to']   ?? date('Y-m-d');

// --- SHOP AWARE ---
if ($user_role === 'admin') {
    $shop_id = $_GET['shop_id'] ?? 0; // 0 = all
} else {
    $shop_id = $user_shop_id;
}

/* ---------------- SQL -------------------- */
$sql = "
    SELECT 
        u.fullname AS cashier,
        COUNT(s.id) AS total_sales,
        SUM(s.total) AS total_amount,
        SUM(s.paid) AS total_paid,
        SUM(s.total - s.paid) AS balance,
        sh.name AS shop_name
    FROM sales s
    JOIN users u ON s.created_by = u.id
    LEFT JOIN shops sh ON s.shop_id = sh.id
    WHERE DATE(s.sale_date) BETWEEN ? AND ?
";

$params = [$from, $to];

if ($shop_id != 0) {
    $sql .= " AND s.shop_id = ? ";
    $params[] = $shop_id;
}

$sql .= "
    GROUP BY u.id, u.fullname, sh.name
    ORDER BY total_amount DESC
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ------------ CSV Export ------------- */
if (isset($_GET['export']) && $_GET['export']==='csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename=sales_by_user_'.$from.'_'.$to.'.csv');
    $out = fopen('php://output','w');
    fputcsv($out, ['Cashier/User','Total Sales','Total Amount','Total Paid','Balance','Shop']);
    foreach ($rows as $r) {
        fputcsv($out, [
            $r['cashier'], 
            $r['total_sales'], 
            $r['total_amount'], 
            $r['total_paid'], 
            $r['balance'],
            $r['shop_name']
        ]);
    }
    fclose($out);
    exit;
}
?>

<style>
@media print {
    body * { visibility: hidden; }
    #print-area, #print-area * { visibility: visible; }
    #print-area { position: absolute; left: 0; top: 0; width: 100%; }
    .no-print { display: none !important; }
}
</style>

<div class="col-md-10 ml-sm-auto px-4 py-4">

<div class="no-print">
<h3 class="text-success">Sales by User</h3>

<form class="form-inline mb-3">

    <label class="mr-2">From</label>
    <input type="date" name="from" value="<?= $from ?>" class="form-control mr-2">

    <label class="mr-2">To</label>
    <input type="date" name="to" value="<?= $to ?>" class="form-control mr-2">

    <?php if ($user_role === 'admin'): ?>
        <label class="mr-2">Shop</label>
        <select name="shop_id" class="form-control mr-2">
            <option value="0">All Shops</option>
            <?php
                $shops = $pdo->query("SELECT id,name FROM shops ORDER BY name")->fetchAll();
                foreach ($shops as $sh):
            ?>
            <option value="<?= $sh['id'] ?>" <?= ($shop_id == $sh['id'] ? 'selected' : '') ?>>
                <?= $sh['name'] ?>
            </option>
            <?php endforeach; ?>
        </select>
    <?php endif; ?>

    <button class="btn btn-primary mr-2">Filter</button>

    <a href="?from=<?= $from ?>&to=<?= $to ?>&shop_id=<?= $shop_id ?>&export=csv" 
       class="btn btn-outline-secondary mr-2">Export CSV</a>

    <button type="button" class="btn btn-dark" onclick="window.print()">Print</button>
</form>
</div>

<!-- PRINT AREA -->
<div id="print-area">

<h4 class="text-center">
    <strong>Sales by User (<?= date('d M Y', strtotime($from)) ?> — <?= date('d M Y', strtotime($to)) ?>)</strong>
</h4>

<?php if ($shop_id == 0): ?>
    <p class="text-center"><em>All Shops</em></p>
<?php else: ?>
    <p class="text-center"><em>Shop: <?= htmlspecialchars($rows[0]['shop_name'] ?? '') ?></em></p>
<?php endif; ?>

<div class="table-responsive">
<table class="table table-bordered table-sm">
    <thead>
        <tr>
            <th>Cashier/User</th>
            <th>Total Sales</th>
            <th>Total Amount</th>
            <th>Total Paid</th>
            <th>Balance</th>
            <th>Shop</th>
        </tr>
    </thead>

    <tbody>
        <?php
        $g_sales = 0;
        $g_amount = 0;
        $g_paid = 0;
        $g_balance = 0;

        foreach ($rows as $r):
            $g_sales += $r['total_sales'];
            $g_amount += $r['total_amount'];
            $g_paid += $r['total_paid'];
            $g_balance += $r['balance'];
        ?>
        <tr>
            <td><?= htmlspecialchars($r['cashier']) ?></td>
            <td><?= $r['total_sales'] ?></td>
            <td><?= number_format($r['total_amount'], 2) ?></td>
            <td><?= number_format($r['total_paid'], 2) ?></td>
            <td><?= number_format($r['balance'], 2) ?></td>
            <td><?= htmlspecialchars($r['shop_name']) ?></td>
        </tr>
        <?php endforeach; ?>
    </tbody>

    <tfoot>
        <tr>
            <th>Totals</th>
            <th><?= $g_sales ?></th>
            <th><?= number_format($g_amount, 2) ?></th>
            <th><?= number_format($g_paid, 2) ?></th>
            <th><?= number_format($g_balance, 2) ?></th>
            <th></th>
        </tr>
    </tfoot>

</table>
</div>

</div> <!-- print area -->

</div> <!-- wrapper -->

<?php require_once '../includes/footer.php'; ?>
