<?php
session_start();
require_once '../assets/db_connect.php';
require_once '../includes/header.php';
require_once '../includes/sidebar.php';

if (!isset($_SESSION['user'])) { 
    echo "Access denied"; 
    exit; 
}

$user_role = $_SESSION['user']['role'];
$user_shop_id = $_SESSION['user']['shop_id'];

$from = $_GET['from'] ?? date('Y-m-01');
$to   = $_GET['to']   ?? date('Y-m-d');

// --- SHOP AWARE ---
if ($user_role === 'admin') {
    $shop_id = $_GET['shop_id'] ?? 0;
} else {
    $shop_id = $user_shop_id;
}

/* -------------------- SQL ---------------------- */
$sql = "
   SELECT 
      p.name AS product,
      SUM(sl.qty) AS total_qty,
      SUM(sl.qty * sl.unit_price - sl.discount) AS revenue,
      sh.name AS shop_name
   FROM sale_lines sl
   JOIN sales s ON sl.sale_id = s.id
   JOIN products p ON sl.product_id = p.id
   LEFT JOIN shops sh ON s.shop_id = sh.id
   WHERE DATE(s.sale_date) BETWEEN ? AND ?
";

$params = [$from, $to];

if ($shop_id != 0) {
    $sql .= " AND s.shop_id = ? ";
    $params[] = $shop_id;
}

$sql .= "
   GROUP BY p.id, p.name, sh.name
   ORDER BY revenue DESC
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* -------- CSV EXPORT -------- */
if (isset($_GET['export']) && $_GET['export']==='csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename=sales_by_product_'.$from.'_'.$to.'.csv');
    $out = fopen('php://output','w');
    fputcsv($out, ['Product','Qty Sold','Revenue','Shop']);
    foreach ($rows as $r) {
        fputcsv($out, [$r['product'], $r['total_qty'], $r['revenue'], $r['shop_name']]);
    }
    fclose($out);
    exit;
}
?>

<style>
@media print {
    body * { visibility: hidden; }
    #print-area, #print-area * { visibility: visible; }
    #print-area { position: absolute; left: 0; top: 0; width: 100%; }
    .no-print { display: none !important; }
}
</style>

<div class="col-md-10 ml-sm-auto px-4 py-4">

<div class="no-print">
<h3 class="text-success">Sales by Product</h3>

<form class="form-inline mb-3">

  <label class="mr-2">From</label>
  <input type="date" name="from" value="<?= $from ?>" class="form-control mr-2">

  <label class="mr-2">To</label>
  <input type="date" name="to" value="<?= $to ?>" class="form-control mr-2">

  <?php if ($user_role === 'admin'): ?>
      <label class="mr-2">Shop</label>
      <select name="shop_id" class="form-control mr-2">
          <option value="0">All Shops</option>
          <?php
              $shops = $pdo->query("SELECT id,name FROM shops ORDER BY name")->fetchAll();
              foreach ($shops as $sh):
          ?>
          <option value="<?= $sh['id'] ?>" <?= ($shop_id == $sh['id'] ? 'selected' : '') ?>>
              <?= $sh['name'] ?>
          </option>
          <?php endforeach; ?>
      </select>
  <?php endif; ?>

  <button class="btn btn-primary mr-2">Filter</button>

  <a href="?from=<?= $from ?>&to=<?= $to ?>&shop_id=<?= $shop_id ?>&export=csv" 
     class="btn btn-outline-secondary mr-2">Export CSV</a>

  <button type="button" class="btn btn-dark" onclick="window.print()">Print</button>
</form>
</div>


<!-- PRINT CONTENT -->
<div id="print-area">
<h4 class="text-center">
    <strong>Product Sales Report (<?= date('d M Y', strtotime($from)) ?> — <?= date('d M Y', strtotime($to)) ?>)</strong>
</h4>

<?php if ($shop_id == 0): ?>
    <p class="text-center"><em>All Shops</em></p>
<?php else: ?>
    <p class="text-center"><em>Shop: <?= htmlspecialchars($rows[0]['shop_name'] ?? '') ?></em></p>
<?php endif; ?>

<div class="table-responsive">
<table class="table table-bordered table-sm">
    <thead>
        <tr>
            <th>Product</th>
            <th>Qty Sold</th>
            <th>Revenue</th>
            <th>Shop</th>
        </tr>
    </thead>

    <tbody>
        <?php 
        $total_qty = 0;
        $total_revenue = 0;

        foreach ($rows as $r): 
            $total_qty += $r['total_qty'];
            $total_revenue += $r['revenue'];
        ?>
        <tr>
            <td><?= htmlspecialchars($r['product']) ?></td>
            <td><?= $r['total_qty'] ?></td>
            <td><?= number_format($r['revenue'],2) ?></td>
            <td><?= htmlspecialchars($r['shop_name']) ?></td>
        </tr>
        <?php endforeach; ?>
    </tbody>

    <tfoot>
        <tr>
            <th>Total</th>
            <th><?= $total_qty ?></th>
            <th><?= number_format($total_revenue,2) ?></th>
            <th></th>
        </tr>
    </tfoot>

</table>
</div>

</div><!-- end print -->

</div><!-- page wrapper -->

<?php require_once '../includes/footer.php'; ?>
