<?php
session_start();
require_once '../assets/db_connect.php';
require_once '../includes/header.php';
require_once '../includes/sidebar.php';

// Ensure user is logged in
if (!isset($_SESSION['user'])) {
    header("Location: ../auth/login.php");
    exit;
}

$user = $_SESSION['user'];
$shop_id = $user['shop_id']; // current shop

$message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Sanitize and trim inputs
    $sku = strtoupper(trim($_POST['sku']));
    $name = trim($_POST['name']);
    $generic_name = trim($_POST['generic_name']);
    $unit = trim($_POST['unit']);
    $category = trim($_POST['category']);
    $vat_percent = floatval($_POST['vat_percent']); // ensure numeric
    $buying_price = floatval($_POST['buying_price']); // ensure numeric

    // Validation
    if (empty($name)) {
        $message = "<div class='alert alert-danger'>Product name is required.</div>";
    } elseif ($vat_percent < 0 || $vat_percent > 100) {
        $message = "<div class='alert alert-danger'>VAT percent must be between 0 and 100.</div>";
    } elseif ($buying_price < 0) {
        $message = "<div class='alert alert-danger'>Buying price cannot be negative.</div>";
    } else {
        // Auto-generate SKU if empty
        if (empty($sku)) {
            $base = strtoupper(substr(preg_replace("/[^a-zA-Z]/", "", $name), 0, 3));
            do {
                $sku = $base . '-' . rand(100, 999);
                $stmt_check = $pdo->prepare("SELECT COUNT(*) FROM products WHERE sku = ?");
                $stmt_check->execute([$sku]);
                $count = $stmt_check->fetchColumn();
            } while ($count > 0); // ensure uniqueness
        } else {
            // Check for existing SKU
            $stmt_check = $pdo->prepare("SELECT COUNT(*) FROM products WHERE sku = ? AND shop_id = ?");
            $stmt_check->execute([$sku, $shop_id]);
            if ($stmt_check->fetchColumn() > 0) {
                $message = "<div class='alert alert-danger'>SKU already exists. Please choose another.</div>";
            }
        }

        // Insert product if no errors so far
        if (empty($message)) {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO products 
                    (sku, name, generic_name, unit, category, vat_percent, buying_price, shop_id)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$sku, $name, $generic_name, $unit, $category, $vat_percent, $buying_price, $shop_id]);

                $message = "<div class='alert alert-success'>Product added successfully! SKU: <strong>$sku</strong></div>";
            } catch (PDOException $e) {
                $message = "<div class='alert alert-danger'>Error adding product: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
        }
    }
}
?>

<div class="col-md-10 ml-sm-auto px-4 py-4">
    <h3 class="text-success mb-4">Add New Product</h3>

    <?= $message ?>

    <form method="POST" action="">
        <div class="form-group mb-3">
            <label>SKU (leave empty to auto-generate)</label>
            <input type="text" name="sku" class="form-control" value="<?= isset($sku) ? htmlspecialchars($sku) : '' ?>">
        </div>
        <div class="form-group mb-3">
            <label>Product Name *</label>
            <input type="text" name="name" class="form-control" required value="<?= isset($name) ? htmlspecialchars($name) : '' ?>">
        </div>
        <div class="form-group mb-3">
            <label>Generic Name</label>
            <input type="text" name="generic_name" class="form-control" value="<?= isset($generic_name) ? htmlspecialchars($generic_name) : '' ?>">
        </div>
        <div class="form-group mb-3">
            <label>Unit</label>
            <input type="text" name="unit" class="form-control" value="<?= isset($unit) ? htmlspecialchars($unit) : 'tablet' ?>">
        </div>
        <div class="form-group mb-3">
            <label>Category</label>
            <input type="text" name="category" class="form-control" value="<?= isset($category) ? htmlspecialchars($category) : '' ?>">
        </div>
        <div class="form-group mb-3">
            <label>VAT %</label>
            <input type="number" step="0.01" min="0" max="100" name="vat_percent" class="form-control" value="<?= isset($vat_percent) ? htmlspecialchars($vat_percent) : '0' ?>">
        </div>
        <div class="form-group mb-3">
            <label>Default Buying Price</label>
            <input type="number" step="0.01" min="0" name="buying_price" class="form-control" value="<?= isset($buying_price) ? htmlspecialchars($buying_price) : '0' ?>">
        </div>
        <button type="submit" class="btn btn-success">Add Product</button>
        <a href="list_products.php" class="btn btn-secondary">Back</a>
    </form>
</div>

<?php require_once '../includes/footer.php'; ?>
