<?php
session_start();
require_once '../assets/db_connect.php';    // adjust path if needed
require_once '../includes/header.php';
require_once '../includes/sidebar.php';

// ensure user logged in
if (!isset($_SESSION['user'])) {
    header("Location: ../auth/login.php");
    exit;
}

$user = $_SESSION['user'];
$user_id = $user['id'] ?? 1;
$shop_id = $user['shop_id'] ?? null;

$message = '';

// Load products (only for this shop or all if you prefer)
try {
    $stmt = $pdo->prepare("SELECT id, name, buying_price, selling_price FROM products WHERE shop_id = ? ORDER BY name ASC");
    $stmt->execute([$shop_id]);
    $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $products = [];
    $message = "<div class='alert alert-danger'>Error loading products: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Load suppliers
try {
    $stmt = $pdo->query("SELECT id, name FROM suppliers ORDER BY name ASC");
    $suppliers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $suppliers = [];
    $message = "<div class='alert alert-danger'>Error loading suppliers.</div>";
}

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id   = intval($_POST['product_id'] ?? 0);
    $batch_no     = trim($_POST['batch_no'] ?? '');
    $qty          = intval($_POST['quantity'] ?? 0);
    $cost_price   = floatval($_POST['buying_price'] ?? 0);
    $sell_price   = floatval($_POST['selling_price'] ?? 0);
    $expiry_date  = !empty($_POST['expiry_date']) ? $_POST['expiry_date'] : null;
    $supplier_id  = !empty($_POST['supplier_id']) ? intval($_POST['supplier_id']) : null;
    $created_by   = $user_id;
    $now_updated_by = $user['fullname'] ?? $user_id;

    // Basic validation
    if ($product_id <= 0) {
        $message = "<div class='alert alert-danger'>Please select a product.</div>";
    } elseif ($qty <= 0) {
        $message = "<div class='alert alert-danger'>Quantity must be greater than zero.</div>";
    } elseif ($cost_price < 0 || $sell_price < 0) {
        $message = "<div class='alert alert-danger'>Prices cannot be negative.</div>";
    } elseif (empty($batch_no)) {
        $message = "<div class='alert alert-danger'>Batch number is required.</div>";
    } else {
        try {
            // Optional: prevent same batch_no for same product and shop (uncomment if desired)
            $chk = $pdo->prepare("SELECT COUNT(*) FROM product_batches WHERE product_id = ? AND batch_no = ? AND shop_id = ?");
            $chk->execute([$product_id, $batch_no, $shop_id]);
            if ($chk->fetchColumn() > 0) {
                $message = "<div class='alert alert-warning'>This batch number already exists for the selected product in this shop.</div>";
            } else {
                // Insert into product_batches
                $ins = $pdo->prepare("
                    INSERT INTO product_batches
                    (product_id, batch_no, qty, cost_price, sell_price, expiry_date, supplier_id, created_by, created_at, updated_by, updated_at, shop_id)
                    VALUES
                    (:product_id, :batch_no, :qty, :cost_price, :sell_price, :expiry_date, :supplier_id, :created_by, CURRENT_TIMESTAMP(), :updated_by, CURRENT_TIMESTAMP(), :shop_id)
                ");

                $ins->execute([
                    ':product_id'  => $product_id,
                    ':batch_no'    => $batch_no,
                    ':qty'         => $qty,
                    ':cost_price'  => number_format($cost_price, 2, '.', ''),
                    ':sell_price'  => number_format($sell_price, 2, '.', ''),
                    ':expiry_date' => $expiry_date,
                    ':supplier_id' => $supplier_id,
                    ':created_by'  => $created_by,
                    ':updated_by'  => $now_updated_by,
                    ':shop_id'     => $shop_id
                ]);

                // Update product default buying_price and selling_price (optional but useful)
                $upd = $pdo->prepare("UPDATE products SET buying_price = :bp, selling_price = :sp WHERE id = :pid AND shop_id = :shop_id");
                $upd->execute([
                    ':bp' => number_format($cost_price, 2, '.', ''),
                    ':sp' => number_format($sell_price, 2, '.', ''),
                    ':pid' => $product_id,
                    ':shop_id' => $shop_id
                ]);

                $message = "<div class='alert alert-success'>Batch added successfully.</div>";

                // Clear POST values to reset the form (so user can add another)
                $_POST = [];
            }
        } catch (PDOException $e) {
            $message = "<div class='alert alert-danger'>DB Error: " . htmlspecialchars($e->getMessage()) . "</div>";
        }
    }
}
?>

<div class="col-md-10 ml-sm-auto px-4 py-4">
    <h3 class="text-success mb-4">Add Product Batch</h3>

    <?= $message ?>

    <form method="POST" action="">
        <div class="form-row">
            <div class="form-group col-md-6">
                <label>Product *</label>
                <select name="product_id" class="form-control" required>
                    <option value="">-- Select Product --</option>
                    <?php foreach ($products as $p): ?>
                        <option value="<?= $p['id'] ?>" <?= (isset($_POST['product_id']) && $_POST['product_id']==$p['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($p['name']) ?> (BP: <?= number_format($p['buying_price'],2) ?>, SP: <?= number_format($p['selling_price'] ?? 0,2) ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group col-md-6">
                <label>Supplier</label>
                <select name="supplier_id" class="form-control">
                    <option value="">-- Select Supplier (optional) --</option>
                    <?php foreach ($suppliers as $s): ?>
                        <option value="<?= $s['id'] ?>" <?= (isset($_POST['supplier_id']) && $_POST['supplier_id']==$s['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($s['name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>

        <div class="form-row">
            <div class="form-group col-md-4">
                <label>Batch No *</label>
                <input type="text" name="batch_no" class="form-control" value="<?= htmlspecialchars($_POST['batch_no'] ?? '') ?>" required>
            </div>

            <div class="form-group col-md-2">
                <label>Quantity *</label>
                <input type="number" name="quantity" class="form-control" min="1" value="<?= htmlspecialchars($_POST['quantity'] ?? 1) ?>" required>
            </div>

            <div class="form-group col-md-3">
                <label>Buying Price (cost_price)</label>
                <input type="number" step="0.01" name="buying_price" class="form-control" value="<?= htmlspecialchars($_POST['buying_price'] ?? '0.00') ?>">
            </div>

            <div class="form-group col-md-3">
                <label>Selling Price (sell_price)</label>
                <input type="number" step="0.01" name="selling_price" class="form-control" value="<?= htmlspecialchars($_POST['selling_price'] ?? '0.00') ?>">
            </div>
        </div>

        <div class="form-row">
            <div class="form-group col-md-3">
                <label>Expiry Date</label>
                <input type="date" name="expiry_date" class="form-control" value="<?= htmlspecialchars($_POST['expiry_date'] ?? '') ?>">
            </div>

            <div class="form-group col-md-3">
                <label>VAT %</label>
                <input type="number" step="0.01" name="vat_percent" class="form-control" value="<?= htmlspecialchars($_POST['vat_percent'] ?? '0.00') ?>">
            </div>

            <div class="form-group col-md-6 d-flex align-items-end">
                <button type="submit" class="btn btn-success mr-2">Add Batch</button>
                <a href="list_batches.php" class="btn btn-secondary">View Batches</a>
            </div>
        </div>
    </form>
</div>

<?php require_once '../includes/footer.php'; ?>
